<?php
namespace App\Http\Controllers;

use App\Models\Donation;
use App\Models\Donor;
use App\Models\Project;
use App\Models\Campaign;
use App\Models\Warehouse;
use App\Models\Delegate;
use App\Models\Beneficiary;
use App\Models\TravelRoute;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class DonationWebController extends Controller
{
    public function index()
    {
        $q = (string) request('q', '');
        $projectId = request('project_id');
        $campaignId = request('campaign_id');
        $cashDonations = Donation::with(['donor', 'project', 'campaign', 'warehouse'])
            ->where('type', 'cash')
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->whereHas('donor', function ($d) use ($q) {
                        $d->where('name', 'like', '%' . $q . '%');
                    })
                        ->orWhere('receipt_number', 'like', '%' . $q . '%')
                        ->orWhereHas('project', function ($p) use ($q) {
                            $p->where('name', 'like', '%' . $q . '%');
                        })
                        ->orWhereHas('campaign', function ($c) use ($q) {
                            $c->where('name', 'like', '%' . $q . '%');
                        });
                });
            })
            ->orderByDesc('received_at')->orderByDesc('id')->paginate(12, ['*'], 'cash_page');

        $inKindDonations = Donation::with(['donor', 'project', 'campaign', 'warehouse'])
            ->where('type', 'in_kind')
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->whereHas('donor', function ($d) use ($q) {
                        $d->where('name', 'like', '%' . $q . '%');
                    })
                        ->orWhereHas('project', function ($p) use ($q) {
                            $p->where('name', 'like', '%' . $q . '%');
                        })
                        ->orWhereHas('campaign', function ($c) use ($q) {
                            $c->where('name', 'like', '%' . $q . '%');
                        })
                        ->orWhereHas('warehouse', function ($wh) use ($q) {
                            $wh->where('name', 'like', '%' . $q . '%');
                        });
                });
            })
            ->orderByDesc('received_at')->orderByDesc('id')->paginate(12, ['*'], 'inkind_page');

        $dailyCashSummary = Donation::select(DB::raw("DATE(received_at) as day"), DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total'))
            ->where('type', 'cash')
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->groupBy(DB::raw("DATE(received_at)"))
            ->orderByDesc(DB::raw("DATE(received_at)"))
            ->limit(14)
            ->get()
            ->map(function ($r) {
                return ['day' => (string) $r->day, 'count' => (int) $r->count, 'total' => (float) $r->total];
            });

        $today = now()->toDateString();
        $byChannelRaw = Donation::select('cash_channel', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total'))
            ->where('type', 'cash')
            ->whereDate('received_at', $today)
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->groupBy('cash_channel')
            ->get();
        $todayByChannel = [
            'cash' => ['count' => 0, 'total' => 0.0],
            'instapay' => ['count' => 0, 'total' => 0.0],
            'vodafone_cash' => ['count' => 0, 'total' => 0.0],
            'delegate' => ['count' => 0, 'total' => 0.0],
        ];
        foreach ($byChannelRaw as $r) {
            $key = $r->cash_channel ?: 'cash';
            if (!isset($todayByChannel[$key])) {
                $todayByChannel[$key] = ['count' => 0, 'total' => 0.0];
            }
            $todayByChannel[$key]['count'] = (int) $r->count;
            $todayByChannel[$key]['total'] = (float) $r->total;
        }
        $inKindToday = Donation::select(DB::raw('COUNT(*) as count'), DB::raw('SUM(estimated_value) as total'))
            ->where('type', 'in_kind')
            ->whereDate('received_at', $today)
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->first();
        // لا نخصم من النقدي لأن قنوات "مندوب" و"مقر المؤسسة" تُخزَّن كقنوات مستقلة

        return view('donations.index', compact('cashDonations', 'inKindDonations', 'dailyCashSummary', 'todayByChannel', 'inKindToday', 'q'));
    }

    public function export(Request $request)
    {
        $q = (string) $request->get('q', '');
        $projectId = $request->get('project_id');
        $campaignId = $request->get('campaign_id');
        $month = $request->get('month');
        $year = $request->get('year');
        $infoType = $request->get('type', 'cash'); // cash or in_kind

        $query = Donation::with(['donor', 'project', 'campaign', 'warehouse'])
            ->where('type', $infoType)
            ->when($projectId, function ($qr) use ($projectId) {
                $qr->where('project_id', $projectId);
            })
            ->when($campaignId, function ($qr) use ($campaignId) {
                $qr->where('campaign_id', $campaignId);
            })
            ->when($month, function ($qr) use ($month) {
                $qr->whereMonth('received_at', $month);
            })
            ->when($year, function ($qr) use ($year) {
                $qr->whereYear('received_at', $year);
            })
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->whereHas('donor', function ($d) use ($q) {
                        $d->where('name', 'like', '%' . $q . '%');
                    })
                        ->orWhere('receipt_number', 'like', '%' . $q . '%')
                        ->orWhereHas('project', function ($p) use ($q) {
                            $p->where('name', 'like', '%' . $q . '%');
                        })
                        ->orWhereHas('campaign', function ($c) use ($q) {
                            $c->where('name', 'like', '%' . $q . '%');
                        });
                });
            })
            ->orderByDesc('received_at');

        $filename = "donations_{$infoType}_" . date('Ymd_His') . ".csv";

        return response()->streamDownload(function () use ($query, $infoType) {
            $handle = fopen('php://output', 'w');
            fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF)); // BOM for Excel

            if ($infoType === 'cash') {
                fputcsv($handle, ['ID', 'المتبرع', 'المبلغ', 'المشروع', 'الحملة', 'التاريخ', 'رقم الإيصال']);
                $query->chunk(100, function ($donations) use ($handle) {
                    foreach ($donations as $d) {
                        fputcsv($handle, [
                            $d->id,
                            $d->donor->name ?? '—',
                            $d->amount,
                            $d->project->name ?? '—',
                            $d->campaign->name ?? '—',
                            $d->received_at->format('Y-m-d'),
                            $d->receipt_number
                        ]);
                    }
                });
            } else {
                fputcsv($handle, ['ID', 'المتبرع', 'القيمة التقديرية', 'المخزن', 'المشروع', 'الحملة', 'التاريخ']);
                $query->chunk(100, function ($donations) use ($handle) {
                    foreach ($donations as $d) {
                        fputcsv($handle, [
                            $d->id,
                            $d->donor->name ?? '—',
                            $d->estimated_value,
                            $d->warehouse->name ?? '—',
                            $d->project->name ?? '—',
                            $d->campaign->name ?? '—',
                            $d->received_at->format('Y-m-d')
                        ]);
                    }
                });
            }
            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function create()
    {
        $donors = Donor::orderBy('name')->get();
        $projects = Project::orderBy('name')->get();
        $campaigns = Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $warehouses = Warehouse::orderBy('name')->get();
        $guestHouses = \App\Models\GuestHouse::where(function ($q) {
            $q->where('location', 'like', '%كفر%')
                ->orWhere('location', 'like', '%طنطا%')
                ->orWhere('name', 'like', '%كفر%')
                ->orWhere('name', 'like', '%طنطا%');
        })->orderBy('name')->get();
        $ghKafr = $guestHouses->first(function ($gh) {
            return (strpos($gh->location, 'كفر') !== false) || (strpos($gh->name, 'كفر') !== false);
        });
        $ghTanta = $guestHouses->first(function ($gh) {
            return (strpos($gh->location, 'طنطا') !== false) || (strpos($gh->name, 'طنطا') !== false);
        });
        $delegates = Delegate::orderBy('name')->get();
        $routes = TravelRoute::orderBy('name')->get();
        $beneficiaries = Beneficiary::select('id', 'full_name')->orderBy('full_name')->get();
        return view('donations.create', compact('donors', 'projects', 'campaigns', 'warehouses', 'delegates', 'routes', 'beneficiaries', 'guestHouses', 'ghKafr', 'ghTanta'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'donor_id' => 'nullable|exists:donors,id|required_without:new_donor_name',
            'new_donor_name' => ['required_without:donor_id', 'string', 'min:3', 'unique:donors,name', 'regex:/^[\p{L}\s\.]+$/u'],
            'new_donor_phone' => ['required_with:new_donor_name', 'string', 'unique:donors,phone', 'regex:/^(01[0125][0-9]{8})$/'],
            'new_donor_address' => 'required_with:new_donor_name|string',
            'new_donor_classification' => 'nullable|in:one_time,recurring',
            'new_donor_cycle' => 'nullable|in:monthly,yearly',
            'type' => 'required|in:cash,in_kind',
            'cash_channel' => 'required_if:type,cash|in:cash,instapay,vodafone_cash,delegate',
            'amount' => 'nullable|numeric',
            'currency' => 'nullable|string',
            'receipt_number' => 'nullable|string|max:64|unique:donations,receipt_number',
            'estimated_value' => 'nullable|numeric',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'delegate_id' => 'nullable|exists:delegates,id',
            'route_id' => 'nullable|exists:travel_routes,id',
            'allocation_note' => 'nullable|string',
            'received_at' => 'nullable|date'
        ], [
            'new_donor_name.required_without' => 'يرجى اختيار متبرع أو إدخال اسم متبرع جديد.',
            'new_donor_name.unique' => 'اسم المتبرع هذا مسجل مسبقاً، يرجى اختياره من القائمة.',
            'new_donor_name.min' => 'اسم المتبرع يجب أن يكون 3 أحرف على الأقل.',
            'new_donor_name.regex' => 'اسم المتبرع يجب أن يحتوي على أحرف فقط.',
            'new_donor_phone.unique' => 'رقم الهاتف هذا مسجل مسبقاً لمتبرع آخر.',
            'new_donor_phone.regex' => 'رقم الهاتف يجب أن يكون رقم مصري صحيح (010, 011, 012, 015).',
            'receipt_number.unique' => 'رقم الإيصال هذا مسجل مسبقاً لتبرع آخر.',
        ]);
        if (empty($data['donor_id']) && !empty($data['new_donor_name'])) {
            $newDonor = Donor::create([
                'name' => $data['new_donor_name'],
                'type' => 'individual',
                'phone' => $data['new_donor_phone'] ?? null,
                'address' => $data['new_donor_address'] ?? null,
                'classification' => $data['new_donor_classification'] ?? 'one_time',
                'recurring_cycle' => $data['new_donor_cycle'] ?? null,
                'active' => true
            ]);
            $data['donor_id'] = $newDonor->id;
        }
        unset($data['new_donor_name'], $data['new_donor_phone'], $data['new_donor_address'], $data['new_donor_classification'], $data['new_donor_cycle']);
        if ($data['type'] === 'cash') {
            if (!isset($data['amount'])) {
                return back()->withErrors(['amount' => 'مطلوب مبلغ للتبرع النقدي']);
            }
            if ((float) $data['amount'] <= 0) {
                return back()->withErrors(['amount' => 'المبلغ يجب أن يكون أكبر من صفر']);
            }
            if (!isset($data['receipt_number']) || trim((string) $data['receipt_number']) === '') {
                return back()->withErrors(['receipt_number' => 'رقم الإيصال مطلوب للتبرع النقدي']);
            }
        } else {
            if (!isset($data['warehouse_id'])) {
                return back()->withErrors(['warehouse_id' => 'مطلوب تحديد المخزن للتبرع العيني']);
            }
            if (!isset($data['estimated_value'])) {
                return back()->withErrors(['estimated_value' => 'مطلوب قيمة تقديرية للتبرع العيني']);
            }
            if ((float) $data['estimated_value'] <= 0) {
                return back()->withErrors(['estimated_value' => 'القيمة التقديرية يجب أن تكون أكبر من صفر']);
            }
        }
        $note = (string) ($data['allocation_note'] ?? '');
        if (empty($data['project_id']) && empty($data['campaign_id']) && empty($data['guest_house_id']) && !str_contains($note, 'sponsorship=')) {
            return back()->withErrors(['project_id' => 'يجب اختيار مشروع أو دار ضيافة أو حملة أو تحديد صدقة جارية أو كفالة للتبرع']);
        }
        if (!\Illuminate\Support\Facades\Schema::hasColumn('donations', 'guest_house_id')) {
            unset($data['guest_house_id']);
        }
        // نحتفظ بالقناة كما هي (delegate/hq/cash/instapay/vodafone_cash)
        $donation = Donation::create($data);
        \App\Services\DonationService::postCreate($donation);
        return redirect()->route('donations.index');
    }

    public function show(Donation $donation)
    {
        $donation->load(['donor', 'project', 'campaign', 'warehouse', 'delegate', 'route']);
        return view('donations.show', compact('donation'));
    }

    public function edit(Donation $donation)
    {
        $donors = Donor::orderBy('name')->get();
        $projects = Project::orderBy('name')->get();
        $campaigns = Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $warehouses = Warehouse::orderBy('name')->get();
        $delegates = Delegate::orderBy('name')->get();
        $routes = TravelRoute::orderBy('name')->get();
        $guestHouses = \App\Models\GuestHouse::where(function ($q) {
            $q->where('location', 'like', '%كفر%')
                ->orWhere('location', 'like', '%طنطا%')
                ->orWhere('name', 'like', '%كفر%')
                ->orWhere('name', 'like', '%طنطا%');
        })->orderBy('name')->get();
        $ghKafr = $guestHouses->first(function ($gh) {
            return (strpos($gh->location, 'كفر') !== false) || (strpos($gh->name, 'كفر') !== false);
        });
        $ghTanta = $guestHouses->first(function ($gh) {
            return (strpos($gh->location, 'طنطا') !== false) || (strpos($gh->name, 'طنطا') !== false);
        });
        $beneficiaries = Beneficiary::select('id', 'full_name')->orderBy('full_name')->get();
        return view('donations.edit', compact('donation', 'donors', 'projects', 'campaigns', 'warehouses', 'delegates', 'routes', 'guestHouses', 'ghKafr', 'ghTanta', 'beneficiaries'));
    }

    public function update(Request $request, Donation $donation)
    {
        $data = $request->validate([
            'type' => 'sometimes|in:cash,in_kind',
            'cash_channel' => 'nullable|in:cash,instapay,vodafone_cash,delegate',
            'amount' => 'nullable|numeric',
            'currency' => 'nullable|string',
            'receipt_number' => 'nullable|string|max:64|unique:donations,receipt_number,' . $donation->id,
            'estimated_value' => 'nullable|numeric',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'delegate_id' => 'nullable|exists:delegates,id',
            'route_id' => 'nullable|exists:travel_routes,id',
            'allocation_note' => 'nullable|string',
            'received_at' => 'nullable|date'
        ]);
        if (!\Illuminate\Support\Facades\Schema::hasColumn('donations', 'guest_house_id')) {
            unset($data['guest_house_id']);
        }
        // نحتفظ بالقناة كما هي في التحديث أيضًا
        if ((($data['type'] ?? $donation->type) === 'cash') && isset($data['amount']) && (float) $data['amount'] <= 0) {
            return back()->withErrors(['amount' => 'المبلغ يجب أن يكون أكبر من صفر']);
        }
        if ((($data['type'] ?? $donation->type) === 'in_kind') && isset($data['estimated_value']) && (float) $data['estimated_value'] <= 0) {
            return back()->withErrors(['estimated_value' => 'القيمة التقديرية يجب أن تكون أكبر من صفر']);
        }
        $donation->update($data);
        return redirect()->route('donations.show', $donation);
    }

    public function destroy(Donation $donation)
    {
        $donation->delete();
        return redirect()->route('donations.index');
    }
}
