<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\User;
use Illuminate\Http\Request;

class EmployeeTaskWebController extends Controller
{
    public function index(Request $request)
    {
        // Tasks assigned to employees
        $query = Task::whereHas('assignee', function($q){
            $q->where('is_employee', true);
        })->with(['assignee','assigner'])->orderByDesc('id');

        if($request->has('assigned_to')){
            $query->where('assigned_to', $request->assigned_to);
        }

        $tasks = $query->paginate(50);
        
        return view('employee_tasks.index', compact('tasks'));
    }

    public function create()
    {
        $users = User::where('is_employee', true)->orderBy('name')->get();
        $projects = \App\Models\Project::orderBy('name')->get();
        $campaigns = \App\Models\Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $guestHouses = \App\Models\GuestHouse::orderBy('name')->get();
        
        return view('employee_tasks.create', compact('users','projects','campaigns','guestHouses'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'required|string',
            'volunteer_activity_name' => 'nullable|string',
            'description' => 'nullable|string',
            'assigned_to' => 'nullable|exists:users,id',
            'assigned_by' => 'nullable|exists:users,id',
            'due_date' => 'nullable|date',
            'status' => 'in:pending,in_progress,done',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'rating' => 'nullable|integer|min:1|max:5',
            'evaluation_notes' => 'nullable|string'
        ]);
        
        $task = Task::create($data);
        return redirect()->route('employee-tasks.show', $task);
    }

    public function show(Task $task)
    {
        return view('employee_tasks.show', ['task' => $task->load(['assignee','assigner','project','campaign','guestHouse'])]);
    }

    public function edit(Task $task)
    {
        $users = User::where('is_employee', true)->orderBy('name')->get();
        $projects = \App\Models\Project::orderBy('name')->get();
        $campaigns = \App\Models\Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $guestHouses = \App\Models\GuestHouse::orderBy('name')->get();
        
        return view('employee_tasks.edit', ['task' => $task, 'users' => $users, 'projects' => $projects, 'campaigns' => $campaigns, 'guestHouses' => $guestHouses]);
    }

    public function update(Request $request, Task $task)
    {
        $data = $request->validate([
            'title' => 'sometimes|string',
            'volunteer_activity_name' => 'nullable|string',
            'description' => 'nullable|string',
            'assigned_to' => 'nullable|exists:users,id',
            'assigned_by' => 'nullable|exists:users,id',
            'due_date' => 'nullable|date',
            'status' => 'in:pending,in_progress,done',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'rating' => 'nullable|integer|min:1|max:5',
            'evaluation_notes' => 'nullable|string'
        ]);
        
        $task->update($data);
        return redirect()->route('employee-tasks.show', $task);
    }

    public function destroy(Task $task)
    {
        $task->delete();
        return redirect()->route('employee-tasks.index');
    }
}
