<?php
namespace App\Http\Controllers;

use App\Models\Project;
use Illuminate\Http\Request;

class ProjectWebController extends Controller
{
    public function index(Request $request)
    {
        $q = (string) $request->get('q', '');
        $status = (string) $request->get('status', '');
        $fixed = $request->has('fixed') ? $request->boolean('fixed') : null;
        $projects = Project::query()
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where('name', 'like', '%' . $q . '%');
            })
            ->when($status !== '', function ($qr) use ($status) {
                $qr->where('status', $status);
            })
            ->when(!is_null($fixed), function ($qr) use ($fixed) {
                $qr->where('fixed', $fixed);
            })
            ->orderBy('name')
            ->paginate(20)
            ->appends(['q' => $q, 'status' => $status] + (!is_null($fixed) ? ['fixed' => $fixed] : []));
        return view('projects.index', compact('projects', 'q', 'status', 'fixed'));
    }

    public function create()
    {
        return view('projects.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'fixed' => 'nullable|boolean',
            'status' => 'required|in:active,archived',
            'description' => 'nullable|string',
        ]);
        $data['fixed'] = $data['fixed'] ?? true;
        $project = Project::create($data);
        return redirect()->route('projects.show', $project);
    }

    public function show(Project $project)
    {
        $donationsCount = \App\Models\Donation::where('project_id', $project->id)->count();
        $cashSum = (float) \App\Models\Donation::where('project_id', $project->id)->where('type', 'cash')->sum('amount');
        $inKindSum = (float) \App\Models\Donation::where('project_id', $project->id)->where('type', 'in_kind')->sum('estimated_value');
        $beneficiariesCount = \App\Models\Beneficiary::where('project_id', $project->id)->count();
        $expensesCount = \App\Models\Expense::where('project_id', $project->id)->count();
        $latestDonations = \App\Models\Donation::where('project_id', $project->id)->orderByDesc('id')->limit(5)->get();
        $latestExpenses = \App\Models\Expense::where('project_id', $project->id)->orderByDesc('id')->limit(5)->get();
        $latestBeneficiaries = \App\Models\Beneficiary::where('project_id', $project->id)->orderByDesc('id')->limit(5)->get();
        $volunteers = \App\Models\User::where('is_volunteer', true)->orderBy('name')->get();
        $projectVolunteers = $project->volunteers()->orderBy('name')->get();
        $campaigns = \App\Models\Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $activities = $project->activities()->get();
        $exhibitions = $activities->where('type', 'exhibition');
        $advertisingDays = $activities->where('type', 'advertising');
        $activitiesRevenue = $activities->sum('revenue');
        $activitiesExpenses = $activities->sum('expenses');

        $donationsTotal = $cashSum + $inKindSum + $activitiesRevenue;
        $expensesTotal = (float) \App\Models\Expense::where('project_id', $project->id)->sum('amount') + $activitiesExpenses;
        $netBalance = $donationsTotal - $expensesTotal;
        $cashPct = $donationsTotal > 0 ? round((($cashSum + $activitiesRevenue) / $donationsTotal) * 100) : 0;
        $monthlyVolunteers = $project->monthlyVolunteers()->with('user')->get();

        $exhibitionsRevenue = $exhibitions->sum('revenue');
        $exhibitionsExpenses = $exhibitions->sum('expenses');

        return view('projects.show', compact('project', 'donationsCount', 'cashSum', 'inKindSum', 'beneficiariesCount', 'expensesCount', 'latestDonations', 'latestExpenses', 'latestBeneficiaries', 'projectVolunteers', 'volunteers', 'cashPct', 'donationsTotal', 'expensesTotal', 'netBalance', 'campaigns', 'monthlyVolunteers', 'exhibitions', 'advertisingDays', 'exhibitionsRevenue', 'exhibitionsExpenses', 'activitiesRevenue'));
    }

    public function attachVolunteer(Request $request, Project $project)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'role' => 'nullable|string',
            'started_at' => 'nullable|date',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'hours' => 'nullable|numeric'
        ]);
        $project->volunteers()->syncWithoutDetaching([
            $data['user_id'] => [
                'role' => $data['role'] ?? null,
                'started_at' => $data['started_at'] ?? null,
                'campaign_id' => $data['campaign_id'] ?? null,
                'hours' => $data['hours'] ?? 0
            ]
        ]);
        return redirect()->route('projects.show', $project);
    }

    public function storeMonthlyVolunteer(Request $request, Project $project)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
            'notes' => 'nullable|string'
        ]);

        $exists = $project->monthlyVolunteers()
            ->where('user_id', $data['user_id'])
            ->where('month', $data['month'])
            ->where('year', $data['year'])
            ->exists();

        if (!$exists) {
            $project->monthlyVolunteers()->create($data);
        }

        return redirect()->route('projects.show', $project);
    }

    public function destroyMonthlyVolunteer(Project $project, \App\Models\ProjectMonthlyVolunteer $monthlyVolunteer)
    {
        $monthlyVolunteer->delete();
        return redirect()->route('projects.show', $project);
    }

    public function storeActivity(Request $request, Project $project)
    {
        $data = $request->validate([
            'type' => 'required|in:exhibition,advertising',
            'activity_date' => 'required|date',
            'location' => 'nullable|string',
            'revenue' => 'nullable|numeric|min:0',
            'description' => 'nullable|string'
        ]);

        $data['revenue'] = $data['revenue'] ?? 0;
        $data['expenses'] = 0; // Always set to 0, expenses will be added from expenses page

        $activity = $project->activities()->create($data);

        // Accounting Logic for Revenue
        if ($data['revenue'] > 0) {
            $revenueAccount = \App\Models\Account::firstOrCreate(
                ['name' => 'donation_exhibition'],
                ['code' => '410', 'type' => 'revenue', 'description' => 'Exhibitions & Activities Revenue']
            );

            // Default Cash Account (assuming 102 as per seeds)
            $cashAccount = \App\Models\Account::where('code', '102')->first();
            if (!$cashAccount) {
                $cashAccount = \App\Models\Account::firstOrCreate(
                    ['code' => '102'],
                    ['name' => 'donation_cash', 'type' => 'asset']
                );
            }

            $entry = \App\Models\JournalEntry::create([
                'date' => $data['activity_date'],
                'description' => 'Revenue from Activity: ' . ($data['description'] ?? $data['type']) . ' - Project: ' . $project->name,
                'entry_type' => 'receipt',
                'branch' => 'main',
                'locked' => false
            ]);

            // Debit Cash
            $entry->lines()->create([
                'account_id' => $cashAccount->id,
                'debit' => $data['revenue'],
                'credit' => 0
            ]);

            // Credit Revenue
            $entry->lines()->create([
                'account_id' => $revenueAccount->id,
                'debit' => 0,
                'credit' => $data['revenue']
            ]);
        }

        return redirect()->route('projects.show', $project)->with('success', 'تم حفظ النشاط وتسجيل الإيراد');
    }

    public function destroyActivity(Project $project, \App\Models\ProjectActivity $activity)
    {
        $activity->delete();
        return redirect()->route('projects.show', $project);
    }

    public function detachVolunteer(Project $project, \App\Models\User $user)
    {
        $project->volunteers()->detach($user->id);
        return redirect()->route('projects.show', $project);
    }

    public function setManager(Project $project, Request $request)
    {
        $data = $request->validate([
            'manager_user_id' => 'nullable|exists:users,id',
            'manager_photo' => 'nullable|image|max:5120'
        ]);

        if ($request->hasFile('manager_photo')) {
            $file = $request->file('manager_photo');
            $filename = time() . '_m_' . $file->getClientOriginalName();
            $file->move(public_path('uploads/managers'), $filename);
            $data['manager_photo_url'] = '/uploads/managers/' . $filename;
        }

        unset($data['manager_photo']);

        $project->update($data);
        return redirect()->route('projects.show', $project);
    }

    public function setDeputy(Project $project, Request $request)
    {
        $data = $request->validate([
            'deputy_user_id' => 'nullable|exists:users,id',
            'deputy_photo' => 'nullable|image|max:5120'
        ]);

        if ($request->hasFile('deputy_photo')) {
            $file = $request->file('deputy_photo');
            $filename = time() . '_d_' . $file->getClientOriginalName();
            $file->move(public_path('uploads/managers'), $filename);
            $data['deputy_photo_url'] = '/uploads/managers/' . $filename;
        }

        unset($data['deputy_photo']);

        $project->update($data);
        return redirect()->route('projects.show', $project);
    }

    public function edit(Project $project)
    {
        return view('projects.edit', compact('project'));
    }

    public function update(Request $request, Project $project)
    {
        $data = $request->validate([
            'name' => 'sometimes|string',
            'fixed' => 'nullable|boolean',
            'status' => 'sometimes|in:active,archived',
            'description' => 'nullable|string',
        ]);
        $project->update($data);
        return redirect()->route('projects.show', $project);
    }

    public function destroy(Project $project)
    {
        $project->delete();
        return redirect()->route('projects.index');
    }
}
