<?php
namespace App\Http\Controllers;

use App\Models\TravelRoute;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Http\Request;

class TravelRouteWebController extends Controller
{
    public function index() {
        $q = trim((string) request()->input('q'));
        $minCities = request()->input('min_cities');
        $maxCities = request()->input('max_cities');
        $hasDelegates = request()->input('has_delegates');
        $hasDonations = request()->input('has_donations');
        $sort = request()->input('sort', 'name');
        $dir = request()->input('dir', 'asc');
        $perPage = (int) (request()->input('per_page') ?? 12);

        $base = TravelRoute::query()
            ->select('travel_routes.*')
            ->when($q !== '', function($qb) use ($q){ $qb->where('name','like',"%$q%"); })
            ->when($minCities !== null && $minCities !== '', function($qb) use ($minCities){ $qb->whereRaw('JSON_LENGTH(cities) >= ?', [(int) $minCities]); })
            ->when($maxCities !== null && $maxCities !== '', function($qb) use ($maxCities){ $qb->whereRaw('JSON_LENGTH(cities) <= ?', [(int) $maxCities]); })
            ->when($hasDelegates !== null && $hasDelegates !== '', function($qb) use ($hasDelegates){
                if ((string) $hasDelegates === '1') { $qb->has('delegates'); }
                if ((string) $hasDelegates === '0') { $qb->doesntHave('delegates'); }
            })
            ->when($hasDonations !== null && $hasDonations !== '', function($qb) use ($hasDonations){
                if ((string) $hasDonations === '1') { $qb->has('donations'); }
                if ((string) $hasDonations === '0') { $qb->doesntHave('donations'); }
            })
            ->withCount(['delegates','donations'])
            ->selectSub(
                DB::table('donations')
                    ->selectRaw("SUM(CASE WHEN type='cash' THEN COALESCE(amount,0) ELSE COALESCE(estimated_value,0) END)")
                    ->whereColumn('donations.route_id','travel_routes.id'),
                'donation_total'
            );

        if ($sort === 'name') { $base->orderBy('name', $dir === 'desc' ? 'desc' : 'asc'); }
        if ($sort === 'cities_count') { $base->orderByRaw('JSON_LENGTH(cities) ' . ($dir === 'desc' ? 'DESC' : 'ASC')); }
        if ($sort === 'delegates_count') { $base->orderBy('delegates_count', $dir === 'desc' ? 'desc' : 'asc'); }
        if ($sort === 'donations_count') { $base->orderBy('donations_count', $dir === 'desc' ? 'desc' : 'asc'); }
        if ($sort === 'donation_total') { $base->orderBy('donation_total', $dir === 'desc' ? 'desc' : 'asc'); }

        $routes = $base->paginate(max($perPage,1))->withQueryString();
        return view('routes.index', compact('routes','q','minCities','maxCities','hasDelegates','hasDonations','sort','dir','perPage'));
    }
    public function create() { return view('routes.create'); }
    public function store(Request $request) { $data = $request->validate(['name' => 'required|string','description' => 'nullable|string']); TravelRoute::create($data); return redirect()->route('travel-routes.index'); }
    public function show(TravelRoute $travel_route) {
        $cities = is_array($travel_route->cities ?? null) ? $travel_route->cities : [];
        $delegates = $travel_route->delegates()->orderBy('name')->get();
        $trips = \App\Models\Donation::with(['donor','delegate'])
            ->where('route_id', $travel_route->id)
            ->orderByDesc('received_at')->orderByDesc('id')
            ->paginate(10);
        return view('routes.show', [
            'route' => $travel_route,
            'travel_route' => $travel_route,
            'cities' => $cities,
            'delegates' => $delegates,
            'trips' => $trips,
        ]);
    }
    public function edit(TravelRoute $travel_route) { return view('routes.edit', ['route' => $travel_route]); }
    public function update(Request $request, TravelRoute $travel_route) { $data = $request->validate(['name' => 'sometimes|string','description' => 'nullable|string']); $travel_route->update($data); return redirect()->route('travel-routes.show',$travel_route); }
    public function addCity(Request $request, TravelRoute $travel_route)
    {
        if (!Schema::hasColumn('travel_routes','cities')) {
            Artisan::call('migrate', ['--force' => true]);
            if (!Schema::hasColumn('travel_routes','cities')) {
                return redirect()->back()->withErrors(['cities' => 'تعذّر إنشاء عمود المدن تلقائيًا. يرجى تشغيل dev/run-migrate ثم المحاولة مجددًا.']);
            }
        }
        $data = $request->validate([
            'name' => 'required|string',
            'fare' => 'required|numeric',
            'fare_currency' => 'required|in:EGP,USD,SAR,EUR,AED'
        ]);
        $cities = is_array($travel_route->cities ?? null) ? $travel_route->cities : [];
        $cities[] = ['name' => $data['name'], 'fare' => (float) $data['fare'], 'currency' => $data['fare_currency']];
        $travel_route->cities = $cities;
        $travel_route->save();
        return redirect()->route('travel-routes.show', $travel_route);
    }
    public function addTrip(Request $request, TravelRoute $travel_route)
    {
        $data = $request->validate([
            'trip_title' => 'required|string',
            'trip_description' => 'nullable|string',
            'trip_date' => 'required|date',
            'donation_type' => 'required|in:cash,in_kind',
            'amount' => 'nullable|numeric',
            'estimated_value' => 'nullable|numeric',
            'donor_name' => 'required|string',
            'donor_phone' => ['required','string','regex:/^[0-9٠-٩\+\-\s]+$/'],
            'delegate_id' => 'nullable|exists:delegates,id',
            'trip_location' => 'nullable|string',
            'trip_city' => 'nullable|string'
        ]);

        if ($data['donation_type'] === 'cash' && empty($data['amount'])) {
            return back()->withErrors(['amount' => 'يجب إدخال المبلغ للتبرع النقدي'])->withInput();
        }
        if ($data['donation_type'] === 'in_kind' && empty($data['estimated_value'])) {
            return back()->withErrors(['estimated_value' => 'يجب إدخال القيمة للتبرع العيني'])->withInput();
        }

        $donor = \App\Models\Donor::query()
            ->when(!empty($data['donor_phone']), function($q) use ($data){ $q->where('phone', $data['donor_phone']); })
            ->where('name', $data['donor_name'])->first();
        if (!$donor) {
            $donor = \App\Models\Donor::create([
                'name' => $data['donor_name'],
                'phone' => $data['donor_phone'] ?? null,
                'type' => 'individual',
                'active' => true
            ]);
        }

        $fare = null;
        $fareCurrency = null;
        $cities = is_array($travel_route->cities ?? null) ? $travel_route->cities : [];
        if (!empty($data['trip_city'])) {
            foreach ($cities as $c) {
                if (($c['name'] ?? '') === $data['trip_city']) { $fare = $c['fare'] ?? null; $fareCurrency = $c['currency'] ?? null; break; }
            }
        }

        $noteParts = [];
        $noteParts[] = 'مشوار: ' . $data['trip_title'];
        if (!empty($data['trip_description'])) { $noteParts[] = 'وصف: ' . $data['trip_description']; }
        if (!empty($data['trip_location'])) { $noteParts[] = 'مكان: ' . $data['trip_location']; }
        if ($data['donation_type'] === 'cash') { $noteParts[] = 'قناة الاستلام: ' . ($data['delegate_id'] ? 'مندوب' : 'المركز'); }
        $allocationNote = implode("\n", $noteParts);

        $donation = new \App\Models\Donation();
        $donation->donor_id = $donor->id;
        $donation->type = $data['donation_type'];
        $donation->cash_channel = null;
        $donation->amount = $data['donation_type'] === 'cash' ? (float) ($data['amount'] ?? 0) : null;
        $donation->estimated_value = $data['donation_type'] === 'in_kind' ? (float) ($data['estimated_value'] ?? 0) : null;
        $donation->delegate_id = $data['delegate_id'] ?? null;
        $donation->route_id = $travel_route->id;
        $donation->allocation_note = $allocationNote;
        $donation->received_at = $data['trip_date'];
        $donation->save();

        return redirect()->route('travel-routes.show', $travel_route)->with('ok', true);
    }
    public function destroy(TravelRoute $travel_route) { $travel_route->delete(); return redirect()->route('travel-routes.index'); }

    public function export(Request $request)
    {
        $q = trim((string) $request->input('q'));
        $rows = TravelRoute::query()
            ->select('travel_routes.*')
            ->when($q !== '', function($qb) use ($q){ $qb->where('name','like',"%$q%"); })
            ->withCount(['delegates','donations'])
            ->selectSub(
                DB::table('donations')
                    ->selectRaw("SUM(CASE WHEN type='cash' THEN COALESCE(amount,0) ELSE COALESCE(estimated_value,0) END)")
                    ->whereColumn('donations.route_id','travel_routes.id'),
                'donation_total'
            )
            ->orderBy('name')
            ->limit(2000)
            ->get();

        $cols = ['id','name','description','cities_count','delegates_count','donations_count','donation_total'];
        $filename = 'travel_routes_'.now()->format('Ymd_His').'.csv';
        $headers = ['Content-Type' => 'text/csv; charset=UTF-8','Content-Disposition' => 'attachment; filename="'.$filename.'"'];
        $content = "\xEF\xBB\xBF";
        $content .= implode(',', $cols)."\n";
        foreach ($rows as $r) {
            $citiesCount = is_array($r->cities ?? null) ? count($r->cities) : (int) (is_string($r->cities ?? null) ? 0 : 0);
            $line = [
                $r->id,
                str_replace(["\r","\n"],' ', (string) $r->name),
                str_replace(["\r","\n"],' ', (string) ($r->description ?? '')),
                $citiesCount,
                $r->delegates_count,
                $r->donations_count,
                number_format((float) ($r->donation_total ?? 0), 2, '.', '')
            ];
            $content .= implode(',', array_map(function($v){ return (string) $v; }, $line))."\n";
        }
        return response($content, 200, $headers);
    }

    public function duplicate(Request $request, TravelRoute $travel_route)
    {
        $nameSuffix = trim((string) ($request->input('suffix') ?? 'نسخة'));
        $new = new TravelRoute();
        $new->name = trim(($travel_route->name ?? '').' '.$nameSuffix);
        $new->description = $travel_route->description;
        $new->cities = $travel_route->cities;
        $new->save();
        return redirect()->route('travel-routes.index');
    }
}
