<?php
namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class UserWebController extends Controller
{
    public function index() {
        // Statistics for Employees
        $totalEmployees = User::where('is_employee', true)->count();
        $activeEmployees = User::where('is_employee', true)->where('active', true)->count();
        
        $projectCounts = User::where('is_employee', true)
            ->whereNotNull('project_id')
            ->select('project_id', \Illuminate\Support\Facades\DB::raw('count(*) as total'))
            ->groupBy('project_id')
            ->pluck('total', 'project_id');
            
        $projectsWithStats = \App\Models\Project::orderBy('name')->get()->map(function($p) use ($projectCounts) {
            $p->employees_count = $projectCounts[$p->id] ?? 0;
            return $p;
        });

        // List all users, excluding volunteers (they are managed in the Volunteers page)
        $users = User::where('is_volunteer', false)->with('roles')->orderBy('name')->paginate(20);
        
        return view('users.index', compact('users', 'totalEmployees', 'activeEmployees', 'projectsWithStats'));
    }
    public function create() { $roles = Role::orderBy('name')->get(); return view('users.create', compact('roles')); }
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',
            'phone' => 'nullable|string',
            'is_employee' => 'boolean',
            'is_volunteer' => 'boolean',
            'active' => 'boolean',
            'roles' => 'array',
            'department' => 'nullable|string',
            'job_title' => 'nullable|string',
            'salary' => 'nullable|numeric',
            'join_date' => 'nullable|date',
            'profile_photo' => 'nullable|image|max:2048'
        ]);
        $data['password'] = Hash::make($data['password']);
        
        if ($request->hasFile('profile_photo')) {
            $data['profile_photo_path'] = $request->file('profile_photo')->store('profile-photos', 'public');
        }
        unset($data['profile_photo']);

        $user = User::create($data);
        if (!empty($data['roles'])) { $user->roles()->sync($data['roles']); }
        return redirect()->route('users.show',$user);
    }
    public function show(User $user) { return view('users.show', compact('user')); }
    public function edit(User $user) { $roles = Role::orderBy('name')->get(); return view('users.edit', compact('user','roles')); }
    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name' => 'sometimes|string',
            'email' => 'sometimes|email|unique:users,email,'.$user->id,
            'password' => 'nullable|string|min:6',
            'phone' => 'nullable|string',
            'is_employee' => 'boolean',
            'is_volunteer' => 'boolean',
            'active' => 'boolean',
            'roles' => 'array',
            'department' => 'nullable|string',
            'job_title' => 'nullable|string',
            'salary' => 'nullable|numeric',
            'join_date' => 'nullable|date',
            'profile_photo' => 'nullable|image|max:2048'
        ]);
        if (!empty($data['password'])) { $data['password'] = Hash::make($data['password']); }
        
        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $data['profile_photo_path'] = $request->file('profile_photo')->store('profile-photos', 'public');
        }
        unset($data['profile_photo']);

        $user->update($data);
        if (isset($data['roles'])) { $user->roles()->sync($data['roles']); }
        return redirect()->route('users.show',$user);
    }
    public function destroy(User $user) { $user->delete(); return redirect()->route('users.index'); }
    public function attachRole(User $user, Role $role) { $user->roles()->syncWithoutDetaching([$role->id]); return back(); }
    public function detachRole(User $user, Role $role) { $user->roles()->detach($role->id); return back(); }
}
